﻿/* Copyright 2016 Intellica Corporation.  

This software is protected by FAR Subpart 27.4 - Rights in Data and Copyrights and 
international treaties.  The software was produced by Intellica Corporation of 
San Antonio, Texas for the Veterans Administration (VA) under Contract Number 
VA118-14-C-0015, Project Number TAC-16-28335 and Project Title, 
'Traumatic Brain Injury Clinical Decision Support (TBI CDS) Implementation'.  
Contract dates: 8 May 2014 - 4 January 2016.  This software was finalized 
and uploaded to the VA's Open Source Electronic Health Record Alliance (OSEHRA) 
on 31 December 2015.  Unauthorized reproduction or distribution of this software 
or any portion of it may result in severe civil and criminal penalties and will 
be prosecuted to the maximum possible extent of the law.

For licensing information contact:  info@intellicacorp.com 
*/
using DataAccess;
using System;
using System.Collections.Generic;
using System.Data;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml;
using System.Xml.Linq;
using Newtonsoft;
using Newtonsoft.Json;
using Newtonsoft.Json.Schema;
using Newtonsoft.Json.Linq;
using ext = Ext.Net;

//setup an alias for all direct method calls on this page
[ext.DirectMethodProxyID(IDMode = ext.DirectMethodProxyIDMode.Alias, Alias = "INSTRUMENTS")]


public partial class ucTBICDSInstrument : System.Web.UI.UserControl
{
    private const long lNORMAL_INSTRUMENT = 0;

//<asp:HiddenField ID="htxtIntakeID" runat="server" />
//<asp:HiddenField ID="htxtCPAID" runat="server" />
//<asp:HiddenField ID="htxtPatPWID" runat="server" />
//<asp:HiddenField ID="htxtPWEvtID" runat="server" />
//<asp:HiddenField ID="htxtPWEvtModID" runat="server" />

    public DateTime IntrumentDateTime
    {
        get
        {
            DateTime dtDate = CDataUtils2.GetNullDate();
            if (Session["IntrumentDateTime"] != null)
            {
                dtDate = (DateTime)Session["IntrumentDateTime"];
            }

            return dtDate;
        }
        set { Session["IntrumentDateTime"] = value; }
    }

    public string TBIInstrumentModuleID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentModuleID"] != null)
            {
                strValue = Session["TBIInstrumentModuleID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentModuleID"] = Convert.ToString(value); }
    }

    public string TBIInstrumentIntakeID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentIntakeID"] != null)
            {
                strValue = Session["TBIInstrumentIntakeID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentIntakeID"] = Convert.ToString(value); }
    }

    public string TBIInstrumentCPAID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentCPAID"] != null)
            {
                strValue = Session["TBIInstrumentCPAID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentCPAID"] = Convert.ToString(value); }
    }

    public string TBIInstrumentPatPWID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentPatPWID"] != null)
            {
                strValue = Session["TBIInstrumentPatPWID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentPatPWID"] = Convert.ToString(value); }
    }

    public string TBIInstrumentPWEvtID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentPWEvtID"] != null)
            {
                strValue = Session["TBIInstrumentPWEvtID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentPWEvtID"] = Convert.ToString(value); }
    }

    public string TBIInstrumentPWEvtModID
    {
        get
        {
            string strValue = "";
            if (Session["TBIInstrumentPWEvtModID"] != null)
            {
                strValue = Session["TBIInstrumentPWEvtModID"].ToString();
            }

            return strValue;
        }
        set { Session["TBIInstrumentPWEvtModID"] = Convert.ToString(value); }
    }

    public bool TBIInstrumentReadOnly
    {
        get
        {
            bool bValue = false;
            if (Session["TBIInstrumentReadOnly"] != null)
            {
                bValue = Convert.ToBoolean(Session["TBIInstrumentReadOnly"]);
            }

            return bValue;
        }
        set { Session["TBIInstrumentReadOnly"] = value; }
    }

    public BaseMaster BaseMstr { set; get; }
    
    protected string m_strPatientDNF { set; get; }
    protected string m_strMDWSSiteID { set; get; }
    public long m_lInstrumentID { set; get; }
    public long m_lItakeID { set; get; }

    protected string m_strSkipPattern { set; get; }
    protected string m_strMasks { set; get; }
    protected string m_strInstrumentResponses { set; get; }
    protected string m_strModuleLogic { set; get; }
    
    /// <summary>
    /// 
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="strPatientDNF"></param>
    /// <param name="strMDWSSiteID"></param>
    /// <returns></returns>
    public bool InitializeControl(string strPatientDNF, string strMDWSSiteID)
    {
        if (!String.IsNullOrEmpty(strPatientDNF) && !String.IsNullOrEmpty(strMDWSSiteID))
        {
            m_strPatientDNF = strPatientDNF;
            m_strMDWSSiteID = strMDWSSiteID;

            aspPnlSaveBtn.Visible = (m_lInstrumentID > 1);

            //if the MID was not passed on by ta query string, get the next module assigned
            if (m_lInstrumentID <= 0)
            {
                m_lInstrumentID = GetNexModule(strPatientDNF, strMDWSSiteID); 
            } 

            if (m_lInstrumentID > 0)
            {
                //render the questionnaire
                GetQuestionnaire(m_lInstrumentID, m_lItakeID);
            }
            else
            {
                return false;
            }
            return true;
        }
        return false;
    }

    /// <summary>
    /// US:5768
    /// Render the questionnaire for the passed in module from a XML string 
    /// </summary>
    /// <param name="lMID"></param>
    protected void GetQuestionnaire(long lMID, long lIntakeID)
    {
        CQuestionnaire q = new CQuestionnaire(BaseMstr);

        m_strSkipPattern = q.GetSkipPattern(lMID);
        m_strMasks = q.GetMask(lMID);

        CInstrumentResponse resp = new CInstrumentResponse(BaseMstr);
        m_strInstrumentResponses = resp.GetInstrumentResponsesJson(lMID, lIntakeID);

        //XmlDocument XmlInstrumentDocument = new XmlDocument();
        //XmlInstrumentDocument.LoadXml(q.GetQuestionnaire(lMID));

        //xmlQuestionnaire.DocumentContent = XmlInstrumentDocument.InnerXml;
        //xmlQuestionnaire.TransformSource = Server.MapPath(@"xslQuestionnaire.xslt");

        XmlReaderSettings settings = new XmlReaderSettings();
        settings.DtdProcessing = DtdProcessing.Prohibit;
        settings.XmlResolver = null;
        XmlReader xmlReader = XmlReader.Create(new StringReader(q.GetQuestionnaire(lMID)), settings);

        xmlReader.Read();

        xmlQuestionnaire.DocumentContent = xmlReader.ReadOuterXml();
        //xmlQuestionnaire.DocumentContent = q.GetQuestionnaire(lMID);
        xmlQuestionnaire.TransformSource = Server.MapPath(@"xslQuestionnaire.xslt");

    }

    /// <summary>
    /// Gets the next patient assigned module if no MID is provided
    /// </summary>
    /// <param name="strPatientDNF"></param>
    /// <param name="strMDWSSiteID"></param>
    /// <returns></returns>
    protected long GetNexModule(string strPatientDNF, string strMDWSSiteID) {
        long lMID = 0;
        CQuestionnaire q = new CQuestionnaire(BaseMstr);
        DataSet ds = q.GetNextModuleDS(strPatientDNF, strMDWSSiteID);
        if(ds != null){
            if(ds.Tables[0].Rows.Count > 0){
                lMID = CDataUtils2.GetDSLongValue(ds, "MID");
            }
        }
        return lMID;
    }

    /// <summary>
    /// US:5966
    /// Inititialize the instruments user control 
    /// with the questions and responses for the selected instrument
    /// </summary>
    /// <param name="strPatientDNF"></param>
    /// <param name="strMDWSSiteID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="lPatPWEventModuleID"></param>
    /// <param name="bReadOnly"></param>
    /// <returns></returns>
    public InstrumentOptions LoadInstrument(string strPatientDNF, string strMDWSSiteID, long lMID, long lIntakeID, long lPatPWEventModuleID, bool bReadOnly) {
        m_lInstrumentID = lMID;
        m_lItakeID = lIntakeID;

        TBIInstrumentModuleID = lMID.ToString();
        TBIInstrumentIntakeID = lIntakeID.ToString();
        TBIInstrumentPWEvtModID = lPatPWEventModuleID.ToString();
        TBIInstrumentReadOnly = bReadOnly;

        InitializeControl(strPatientDNF, strMDWSSiteID); 

        return new InstrumentOptions
        {
            SkipPattern = m_strSkipPattern,
            TextMask = m_strMasks,
            ResponsesData = m_strInstrumentResponses
        };
    }

    /// <summary>
    /// US:5768
    /// Iterates the submitted responses and load the necessary variables to write to DB and calculate scores.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected List<CInstrumentResponse.IntakeResponse> GetInstrumentResponses()
    {
        List<CInstrumentResponse.IntakeResponse> ResponsesCollection = new List<CInstrumentResponse.IntakeResponse>{};

        foreach (string key in Request.Form)
        {
            if (key.StartsWith("grp"))
            {
                string strResponse = Request.Form[key].ToString();
                string[] strRespComponents = strResponse.Split('|');

                string[] strName = key.Split('_');
                long lMID = Convert.ToInt32(strName[2]);
                long lTID = Convert.ToInt32(strName[3]);
                long lQID = Convert.ToInt32(strName[4]);
                long lRID = 0;
                if (strName.GetLength(0) > 5)
                {
                    lRID = Convert.ToInt32(strName[5]);
                }

                long lResponseType = 0;
                if (strName.GetLength(0) > 6)
                {
                    lResponseType = Convert.ToInt32(strName[6]);
                }

                string strIdentifier = String.Empty;
                if (strName.GetLength(0) > 7)
                {
                    strIdentifier = strName[7];
                }


                string strRespText = String.Empty;
                string strRespUnit = String.Empty;
                decimal dScoreValue = 0;

                //check if the answer is a checkbox
                //(checkboxes come along with multiple responses comma separated)
                bool isCheckbox = false;

                bool bByPassResponse = false;

                //get radio buttons controls
                #region RADIO
                if (key.ToLower().IndexOf("grpradio_") >= 0)
                {
                    strRespText = strRespComponents[0];
                    lRID = Convert.ToInt32(strRespComponents[1]);
                    lResponseType = Convert.ToInt32(strRespComponents[3]);
                    strIdentifier = strRespComponents[4];
                    if (strRespComponents[2] != "")
                    {
                        dScoreValue = Convert.ToDecimal(strRespComponents[2]);
                    }
                }
                #endregion

                //get checkboxes controls
                #region CHECKBOX
                if (key.ToLower().IndexOf("grpcheck_") >= 0)
                {
                    isCheckbox = true;

                    // loop the response for multiple selections
                    string[] strRespCollection = strResponse.Split(',');
                    foreach (string strR in strRespCollection)
                    {
                        // In order to avoid wrong braking of the value string,
                        // any commas in the response value text was changed to this pattern **"**.
                        // Now, that the value string was properly divided, it can be switched back to a regular comma.

                        string strRespTextReplaced = Regex.Replace(strR.Split('|')[0], "\\*\\*\"\\*\\*", ",");

                        strRespText = strRespTextReplaced;
                        lRID = Convert.ToInt32(strR.Split('|')[1]);
                        lResponseType = Convert.ToInt32(strR.Split('|')[3]);
                        strIdentifier = strR.Split('|')[4];
                        if (strR.Split('|')[2].Length > 0)
                        {
                            dScoreValue = Convert.ToDecimal(strR.Split('|')[2]);
                        }

                        // call code here to write the current checkbox
                        ResponsesCollection.Add(new CInstrumentResponse.IntakeResponse
                        { 
                            PatientID = BaseMstr.SelectedPatientID,
                            MID = lMID,
                            TID = lTID,
                            QID = lQID,
                            RID = lRID,
                            Identifier = strIdentifier,
                            ScoreValue = dScoreValue,
                            ResponseValue = strRespText,
                            ResponseType = lResponseType
                        });

                    }
                }
                #endregion

                //get combo controls
                #region COMBO
                if (key.ToLower().IndexOf("grpselect_") >= 0)
                {
                    strResponse = Request.Form[key].ToString();
                    if (strResponse.Split('|').Length > 1)
                    {
                        dScoreValue = 0;
                        strRespText = strResponse.Split('|')[0];
                        decimal.TryParse(strResponse.Split('|')[1], out dScoreValue);
                    }
                    else {
                        bByPassResponse = true;
                    }
                }
                #endregion

                //get textbox controls
                #region TEXTBOX
                if (key.ToLower().IndexOf("grpctrltext_") >= 0)
                {
                    strRespText = Request.Form[key].ToString();
                    if (strRespText.Length < 1) {
                        bByPassResponse = true;
                    }
                }
                #endregion

                //get TITLE control
                #region TITLE
                if (key.ToLower().IndexOf("grptitle_") >= 0)
                {
                    strRespText = Request.Form[key].ToString();
                    if (strRespText.Replace("\r\n","").Trim().Length < 1)
                    {
                        bByPassResponse = true;
                    }
                }
                #endregion

                // These values were collected from the response
                // ----------------------------------------------------
                // lMID - module id
                // lTID - topic id
                // lQID - question id
                // lRID - response id
                // lScoreValue - response score value
                // strRespText - text of the response
                // strRespUnit - response unit (usually for text boxes)
                // 

                if (!isCheckbox && !bByPassResponse)
                {
                    ResponsesCollection.Add(new CInstrumentResponse.IntakeResponse
                    {
                        PatientID = BaseMstr.SelectedPatientID,
                        MID = lMID,
                        TID = lTID,
                        QID = lQID,
                        RID = lRID,
                        Identifier = strIdentifier,
                        ScoreValue = dScoreValue,
                        ResponseValue = strRespText,
                        ResponseType = lResponseType
                    });
                }
            }
        }


        return ResponsesCollection;
    }

    [ext.DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// US:6247
    /// Renders and shows the completed instrument report on screen
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    public void UpdateInstrumentFields( string strintake_id,
                                        string strpat_pw_id,
                                        string strpat_pw_event_id,
                                        string strpw_event_module_id)
    {

        TBIInstrumentIntakeID = strintake_id;
        TBIInstrumentPatPWID = strpat_pw_id;
        TBIInstrumentPWEvtID = strpat_pw_event_id;
        TBIInstrumentPWEvtModID = strpw_event_module_id;
    }


    [ext.DirectMethod(ShowMask = true, Msg = "<span role='alert'>Please wait.</span>", Timeout = 60000)]
    /// <summary>
    /// US:5768 US:5862
    /// Submit the responses collection to the DB.    
    /// </summary>
    /// <returns></returns>
    public AddInstrumentResponsesResult AddInstrumentResponses(string strAssessmentDate, 
                                                               string strAssessmentTime)
    {
        bool bResult = true;
        string strMsgTitle = String.Empty;
        string strMessage = String.Empty;

        CIntake intake = new CIntake(BaseMstr);

        List<CInstrumentResponse.IntakeResponse> responses = this.GetInstrumentResponses();
        CInstrumentResponse InstrumentResponse = new CInstrumentResponse(BaseMstr); 

        long lIntakeID = 0;
        long.TryParse(TBIInstrumentIntakeID, out lIntakeID);

        string strPatientID = BaseMstr.SelectedPatientID; ;
        long lMID = 0;

        //validate if instrument is read-only...
        if (IsReadOnly())
        {
            //instrumet is READ_ONLY; exit...
            bResult = false;
            strMsgTitle = "Please Review";
            strMessage = "Instrument is Read-Only and cannot be updated.";

            return new AddInstrumentResponsesResult
            {
                saved = bResult,
                patient_id = BaseMstr.SelectedPatientID,
                mid = 0,
                intake_id = lIntakeID,
                cpa_id = 0,
                pat_pw_id = 0,
                pat_pw_event_id = 0,
                pw_event_module_id = 0,
                score_error = false,
                score_error_msg = "",
                title = strMsgTitle,
                message = strMessage,
                status_code = 1
            };
        }

        //validate responses integrity...
        if (!ValidateResponsesIntegrity(responses))
        {
            //instrumet is READ_ONLY; exit...
            bResult = false;
            strMsgTitle = "Responses Integrity Exception";
            strMessage = "One or more responses do not correspond to this instrument.";

            return new AddInstrumentResponsesResult
            {
                saved = bResult,
                patient_id = BaseMstr.SelectedPatientID,
                mid = 0,
                intake_id = lIntakeID,
                cpa_id = 0,
                pat_pw_id = 0,
                pat_pw_event_id = 0,
                pw_event_module_id = 0,
                score_error = false,
                score_error_msg = "",
                title = strMsgTitle,
                message = strMessage,
                status_code = 1
            };
        }

        //clear all previous responses...
        if (responses.Count > 0)
        {
            lMID = responses[0].MID;
            RemoveAllResponses(lIntakeID, lMID);
        }
        else
        {
            //there are no responses; exit...
            bResult = false;
            strMsgTitle = "Please Review";
            strMessage = "No changes or responses where found.";

            return new AddInstrumentResponsesResult
            {
                saved = bResult,
                patient_id = BaseMstr.SelectedPatientID,
                mid = 0,
                intake_id = lIntakeID,
                cpa_id = 0,
                pat_pw_id = 0,
                pat_pw_event_id = 0,
                pw_event_module_id = 0,
                score_error = false,
                score_error_msg = "",
                title = strMsgTitle,
                message = strMessage,
                status_code = 1
            };

        }

        
        //Get a NEW INTAKE_ID or reuse the one we already have
        if (lIntakeID <= 0)
        {
            lIntakeID = intake.GetNextIntakeID(BaseMstr, BaseMstr.SelectedPatientID, lMID);
            TBIInstrumentIntakeID = lIntakeID.ToString();
        }
                
        // ------------------------------------------------------------------------
        // start module 
        if (!intake.StartModule(strPatientID, lMID, 1, lIntakeID))
        {
            //unable to start module...
            bResult = false;
            strMsgTitle = "Error";
            strMessage = "Unable to start module.";

            return new AddInstrumentResponsesResult
            {
                saved = bResult,
                patient_id = BaseMstr.SelectedPatientID,
                mid = 0,
                intake_id = lIntakeID,
                cpa_id = 0,
                pat_pw_id = 0,
                pat_pw_event_id = 0,
                pw_event_module_id = 0,
                score_error = false,
                score_error_msg = "",
                title = strMsgTitle,
                message = strMessage,
                status_code = 1
            };
        }


        // delete responses from a the same MID, Intake_ID not 
        // present in this new submit
        //RemovePrevResponses(responses, lIntakeID);

        //get a dataset of the existing responses for the current instrument intake
        //DataSet dsCurrReponses = null;

        //iterate the responses list
        foreach (CInstrumentResponse.IntakeResponse resp in responses)
        {
            //lMID = resp.MID;

            //if (lIntakeID <= 0)
            //{
            //    CIntake intk = new CIntake();
            //    lIntakeID = intk.GetNextIntakeID(BaseMstr, BaseMstr.SelectedPatientID, lMID);
            //    TBIInstrumentIntakeID = lIntakeID.ToString();
            //}

            //if (dsCurrReponses == null) {
            //    dsCurrReponses = InstrumentResponse.GetInstrumentResponsesDS(lMID, lIntakeID);
            //}

            //check if the response needs to be inserted/overwritten 
            //if (DoInsertResponse(resp, lIntakeID, dsCurrReponses))
            //{
                //write the response in the DB
                if (!InstrumentResponse.InsertInstrumentResponse(resp.PatientID,
                                            lIntakeID,
                                            resp.MID,
                                            resp.TID,
                                            resp.QID,
                                            resp.RID,
                                            (double) resp.ScoreValue,
                                            resp.ResponseValue))
                {
                    bResult = false;
                    strMsgTitle = "ERROR";
                    strMessage = "There was an error while trying to write to the database.";
                } 
            //}
        }


        long lCPA = 0;
        long lPatPWID = 0;
        long lPatPWEvtID = 0;
        long lPwEvtModID = 0;

        long.TryParse(TBIInstrumentCPAID, out lCPA);
        long.TryParse(TBIInstrumentPatPWID, out lPatPWID);
        long.TryParse(TBIInstrumentPWEvtID, out lPatPWEvtID);
        long.TryParse(TBIInstrumentPWEvtModID, out lPwEvtModID);

        // ------------------------------------------------------------------------
        // get logic from the intake_module table.  If logic exist, then process
        string strInstrumentScoreLogic = InstrumentResponse.GetScoreLogic(lMID);
        bool bScoreLogicError = false;
        string strRetLogicError = String.Empty;
        //string strScoreErr = String.Empty;

        CIQEngine iqe = new CIQEngine(BaseMstr);

        if (!String.IsNullOrEmpty(strInstrumentScoreLogic))
        {

            // ------------------------------------------------------------------------
            // instantiate the COM component which will perform the logic calculations


            bool bIQError = false;
            if (iqe == null)
            {
                bIQError = true;
            }

            // ------------------------------------------------------------------------
            // get and load logic vars and send to the COM component

            DataSet dsLV = InstrumentResponse.GetIntakeLogicVarDS(strPatientID, lMID);
            if (bIQError == false)
            {
                iqe.AddSQLVarItem(dsLV);
            }

            // ------------------------------------------------------------------------
            // get and load the responses dataset to the COM component

            //DataSet dsResp = InstrumentResponse.ConvertResponsesToDataSet(responses);
            DataSet dsResp = InstrumentResponse.GetMergedInstrumentResponsesDS(BaseMstr.SelectedPatientID, lMID, lIntakeID);
            if (bIQError == false)
            {
                iqe.AddResponseVarItem(dsResp);
            }

            // ------------------------------------------------------------------------
            // send module logic to the COM component
            if (bIQError == false)
            {
                if (iqe.Calculate(strInstrumentScoreLogic) == false)
                {
                    bScoreLogicError = true;
                    strRetLogicError = iqe.m_strErrorMessage;
                }

                if (!bScoreLogicError)
                {
                    string strJSon = "";
                    //strJSon = iqe.GetJSon(iqe.nJSON_SCORE_TYPE);
                    //if (strJSon.Length > 0)
                    //{
                    //    //process JSON
                    //    this.InsertInstrumentScores(lMID, lIntakeID, strJSon);
                    //}

                     strJSon = iqe.GetJSon(iqe.GetJSON_SCORE_TYPE());
                    if (strJSon.Length > 0)
                    {
                        this.InsertInstrumentScores(lMID, lIntakeID, strJSon);
                    }

                    strJSon = iqe.GetJSon(iqe.GetJSON_FLAG_TYPE());
                    if (strJSon.Length > 0)
                    {
                        this.InsertInstrumentScores(lMID, lIntakeID, strJSon);
                    }

                    strJSon = iqe.GetJSon(iqe.GetJSON_PROB_TYPE());
                    if (strJSon.Length > 0)
                    {
                        this.InsertInstrumentScores(lMID, lIntakeID, strJSon);
                    }
                }
            }
        }

        // ------------------------------------------------------------------------
        if (bScoreLogicError)
        {
            if (intake.DeleteIntake(BaseMstr.SelectedPatientID, lMID, lIntakeID, lPwEvtModID))
            {
                bResult = false;
                strMsgTitle = "Error";
                strMessage = "Unable to reset module.";

                return new AddInstrumentResponsesResult
                {
                    saved = bResult,
                    patient_id = BaseMstr.SelectedPatientID,
                    mid = lMID,
                    intake_id = lIntakeID,
                    cpa_id = lCPA,
                    pat_pw_id = lPatPWID,
                    pat_pw_event_id = lPatPWEvtID,
                    pw_event_module_id = lPwEvtModID,
                    score_error = bScoreLogicError,
                    score_error_msg = "The application has encountered an issue with this instrument and cannot save your responses. Please retake the instrument.~#(br/)#~~#(br/)#~When reporting this error please include the following information:~#(br/)#~" + strRetLogicError,
                    title = strMsgTitle,
                    message = strMessage,
                    status_code = 2
                };
            }
            else
            {
                //unable to start module...
                bResult = false;
                strMsgTitle = "Error";
                strMessage = "Unable to reset module.";

                return new AddInstrumentResponsesResult
                {
                    saved = bResult,
                    patient_id = BaseMstr.SelectedPatientID,
                    mid = 0,
                    intake_id = lIntakeID,
                    cpa_id = 0,
                    pat_pw_id = 0,
                    pat_pw_event_id = 0,
                    pw_event_module_id = 0,
                    score_error = false,
                    score_error_msg = "",
                    title = strMsgTitle,
                    message = strMessage,
                    status_code = 1
                };
            }
        }

        // ------------------------------------------------------------------------
        // mark module as completed 
        intake.CompleteModule(strPatientID, lMID, 1, lIntakeID);

        //update the patient pathway event module table
        CCPA ccpa = new CCPA();
        if (bResult)
        {
            if (lPwEvtModID <= 0)
            {
                bResult = ccpa.UpdatePatPWEventModule(BaseMstr,
                                                BaseMstr.SelectedPatientID,
                                                lPatPWID,
                                                lPatPWEvtID,
                                                lMID,
                                                lIntakeID);

            }
            else
            {
                bResult = ccpa.UpdatePatPWEventModule(BaseMstr,
                                                BaseMstr.SelectedPatientID,
                                                lPatPWID,
                                                lPatPWEvtID,
                                                lMID,
                                                lIntakeID,
                                                lPwEvtModID);
            }
        }
        
        //all done so now update the assessment date based on the date passed in
        //strAssessmentDate   "\"2015-09-09T00:00:00"
        //strAssessmentTime   "\"2008-01-01T01:45:00"
        if (!String.IsNullOrEmpty(strAssessmentDate))
        {
            string strYYYY = "";
            string strMM = "";
            string strDD = "";
            if (strAssessmentDate.Length >= 10)
            {
                strYYYY = strAssessmentDate.Substring(1, 4);
                strMM = strAssessmentDate.Substring(6, 2);
                strDD = strAssessmentDate.Substring(9, 2);
            }

            string strHours = "0";
            string strMinutes = "0";
            string strSeconds = "0";
            if (strAssessmentTime.Length >= 19)
            {
                strHours = strAssessmentTime.Substring(12, 2);
                strMinutes = strAssessmentTime.Substring(15, 2);
                strSeconds = strAssessmentTime.Substring(18, 2);
            }

            DateTime dtRetroDate = CDataUtils2.GetDate(strMM + "/" + strDD + "/" + strYYYY,
                                                       Convert.ToInt32(strHours),
                                                       Convert.ToInt32(strMinutes),
                                                       Convert.ToInt32(strSeconds));

            //update the pathway and intrument date to dtRetroDate
            //PAT_PW_EVENT_MODULE   DATE_STARTED
            //                      DATE_COMPLETED
            //
            //DATA_INTAKE   START_DATE
            //              COMPLETE_DATE
            CPWEvent evt = new CPWEvent();
            evt.UpdateInstrumentDate(BaseMstr, CDataUtils2.ToLong(TBIInstrumentIntakeID), dtRetroDate);

        }


        //update the outcomes screen
        ext.MessageBus.Default.Publish("MBL_RELOADOUTCOMES");

        return new AddInstrumentResponsesResult
        {
            saved = bResult,
            patient_id = BaseMstr.SelectedPatientID,
            mid = lMID,
            intake_id = lIntakeID,
            cpa_id = lCPA,
            pat_pw_id = lPatPWID,
            pat_pw_event_id = lPatPWEvtID,
            pw_event_module_id = lPwEvtModID,
            score_error = bScoreLogicError,
            score_error_msg = strRetLogicError,
            title = strMsgTitle,
            message = strMessage,
            status_code = 0
        };
    }

    /// <summary>
    /// Check if the event is read-only.
    /// If the Event was submitted to TIU then is LOCK and is Read-Only.
    /// </summary>
    /// <returns></returns>
    protected bool IsReadOnly()
    {
        bool bReadOnly = false;

        //validate if instrument is read-only...
        if (TBIInstrumentReadOnly)
        {
            bReadOnly = true;
        }
        else
        {
            //check if event is locked only for normal instruments...
            CIntake intake = new CIntake();
            if (intake.GetModuleType(BaseMstr, Convert.ToInt64(TBIInstrumentModuleID)) == lNORMAL_INSTRUMENT)
            {
                long lPatPWID = 0;
                long lPatPWEvtID = 0;

                long.TryParse(TBIInstrumentPatPWID, out lPatPWID);
                long.TryParse(TBIInstrumentPWEvtID, out lPatPWEvtID);

                CCPA ccpa = new CCPA();
                bReadOnly = ccpa.IsPatPWEventReadOnly(BaseMstr,
                                                      BaseMstr.SelectedPatientID,
                                                      lPatPWID,
                                                      lPatPWEvtID);
            }
        }

        return bReadOnly;

    }

    /// <summary>
    /// Validate Responses Integrity.
    /// Check that all responses correspond to the current MID.
    /// </summary>
    /// <param name="responses"></param>
    /// <returns></returns>
    protected bool ValidateResponsesIntegrity(List<CInstrumentResponse.IntakeResponse> responses)
    {
        long lMID = 0;
        long.TryParse(TBIInstrumentModuleID, out lMID);

        foreach (CInstrumentResponse.IntakeResponse resp in responses)
        {
            if (resp.MID != lMID)
            {
                return false;
            }
        }

        return true;
    }

    ///// <summary>
    ///// 
    ///// </summary>
    ///// <param name="resp"></param>
    ///// <param name="dsCurrResp"></param>
    ///// <returns></returns>
    //public bool DoInsertResponse(CInstrumentResponse.IntakeResponse resp, long lIntakeID, DataSet dsCurrResp)
    //{
    //    bool bInsert = true;
        
    //    if (dsCurrResp != null) {
    //        if (dsCurrResp.Tables[0].Rows.Count > 0) { 
    //            //TODO: (DAVID)
    //        }
    //    }

    //    return bInsert;
    //}

    /// <summary>
    /// Insert Instrument Scores
    /// </summary>
    /// <param name="lMID"></param>
    /// <param name="lIntakeID"></param>
    /// <param name="strJsonScore"></param>
    /// <returns></returns>
    protected bool InsertInstrumentScores(long lMID, long lIntakeID, string strJsonScore)
    {

        //validate the JSON string
        string schemaJson = @"{
                                'type': 'object',
                                'properties': {
                                    'JSCORE': {
                                        'type': 'array',
                                        'required': true,
                                        'items': {
                                            'type': 'object',
                                            'properties': {
                                                'score': {
                                                    'type': 'number',
                                                    'required': true
                                                },
                                                'series': {
                                                    'type': 'number',
                                                    'required': true
                                                },
                                                'interpretation': {
                                                    'type': 'string',
                                                    'required': true
                                                },
                                                'description': {
                                                    'type': 'string',
                                                    'required': true
                                                },
                                                'severity': {
                                                    'type': 'string',
                                                    'required': true
                                                }
                                            }
                                        }
                                    }
                                },
                                'additionalProperties': false
                            }";

        if (strJsonScore.Length > 1)
        {
            try
            {
                JsonSchema schema = JsonSchema.Parse(schemaJson);
                JObject jScore = JObject.Parse(strJsonScore);
                bool bValidJson = jScore.IsValid(schema);
                if (bValidJson)
                {
                    //convert json score string into a specific DataSet type
                    //and avoid the JsonConvert.DeserializeObject<DataSet>(strJsonScore) to specify
                    //the datatype of the score. If the first score is an Integer it will round any decimal.
                    DataSet ds = new DataSet();
                    DataTable dtScore = new DataTable();
                    dtScore.Columns.Add("series", typeof(long));
                    dtScore.Columns.Add("score", typeof(decimal));
                    dtScore.Columns.Add("interpretation", typeof(string));
                    dtScore.Columns.Add("description", typeof(string));
                    dtScore.Columns.Add("severity", typeof(string));

                    IList<JToken> jtokenList = jScore["JSCORE"].Children().ToList();
                    foreach (JToken t in jtokenList)
                    {
                        JScoreRow r = JsonConvert.DeserializeObject<JScoreRow>(t.ToString());
                        dtScore.Rows.Add(r.series, r.score, r.interpretation, r.description, r.severity);
                    }

                    ds.Tables.Add(dtScore);

                    //DataSet ds = JsonConvert.DeserializeObject<DataSet>(strJsonScore);
                    if (ds != null)
                    {
                        CInstrumentResponse instr = new CInstrumentResponse(BaseMstr);

                        //remove previous scores
                        instr.DeleteInstrumentScore(BaseMstr.SelectedPatientID, lMID, lIntakeID);

                        foreach (DataTable dt in ds.Tables)
                        {
                            foreach (DataRow dr in dt.Rows)
                            {
                                //score
                                bool bScoreIsLong = false; //score value could be negative. We need to know if it's LONG or DECIMAL.
                                long lScore = -99999;
                                Decimal dScore = -99999;

                                if (!dr.IsNull("score"))
                                {
                                    if (long.TryParse(dr["score"].ToString(), out lScore) == true)
                                    {
                                        bScoreIsLong = true;
                                    }
                                    else if (decimal.TryParse(dr["score"].ToString(), out dScore) == true)
                                    {
                                        bScoreIsLong = false;
                                    }
                                    else
                                    {
                                    }
                                    
                                }

                                //series
                                long lSeries = -1;
                                if (!dr.IsNull("series"))
                                {
                                    long.TryParse(dr["series"].ToString(), out lSeries);
                                }

                                //interpret_id
                                string strInterpret = String.Empty; ;
                                if (!dr.IsNull("Interpretation"))
                                {
                                    strInterpret = dr["Interpretation"].ToString();
                                }

                                //description
                                string strDescription = String.Empty;
                                if (dr["description"] != null)
                                {
                                    if (!dr.IsNull("description"))
                                    {
                                        strDescription = dr["description"].ToString();
                                    }
                                }

                                //severity
                                long lSeverity = -1;
                                if (!dr.IsNull("severity"))
                                {
                                    long.TryParse(dr["severity"].ToString(), out lSeverity);
                                }

                                //Insert Score
                                if(bScoreIsLong == true)
                                {
                                    //Score is LONG
                                    if (instr.InsertInstrumentScore(BaseMstr.SelectedPatientID, lMID, lIntakeID, lSeries, lScore, strInterpret, strDescription, lSeverity) == false)
                                    {
                                        return false;
                                    }
                                }
                                else //Score if DECIMAL
                                {
                                    if (instr.InsertInstrumentScore(BaseMstr.SelectedPatientID, lMID, lIntakeID, lSeries, (double)dScore, strInterpret, strDescription, lSeverity) == false)
                                    {
                                        return false;
                                    }
                                }

                            }
                        }
                    }
                }
            }
            catch (Exception)
            {
                return false;
            }
        }

        return true;
    }

    /// <summary>
    /// Remove ALL responses
    /// </summary>
    /// <param name="lIntakeID"></param>
    /// <param name="lMID"></param>
    /// <returns></returns>
    public bool RemoveAllResponses(long lIntakeID,
                                   long lMID)
    {
        CInstrumentResponse resp = new CInstrumentResponse(BaseMstr);
        return resp.RemoveAllResponses(BaseMstr.SelectedPatientID, lMID, lIntakeID);   
    }

    /// <summary>
    /// Remove responses from previous time submitted
    /// not present in a new submittal of the instrument
    /// </summary>
    /// <param name="responses"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public bool RemovePrevResponses(List<CInstrumentResponse.IntakeResponse> responses, long lIntakeID) {
        string strRID = String.Empty;
        long lMID = 0;

        //generate CSV string of RID's
        foreach (CInstrumentResponse.IntakeResponse resp in responses)
        {
            lMID = resp.MID;
            strRID += resp.RID + ",";
        }

        //clear trailing comma
        if(strRID.LastIndexOf(",") > 0){
            strRID = strRID.Substring(0, strRID.Length - 1);
        }

        if (strRID.Length > 0) { 
            //send the CSV of RID's to be removed from the intake
            CInstrumentResponse resp = new CInstrumentResponse(BaseMstr);
           return resp.RemovePrevResponses(BaseMstr.SelectedPatientID, lMID, lIntakeID, strRID);
        }

        return true;
    }

    /// <summary>
    /// Removes a specific RID from the intrument intake
    /// </summary>
    /// <param name="response"></param>
    /// <param name="lIntakeID"></param>
    /// <returns></returns>
    public bool RemoveResponse(CInstrumentResponse.IntakeResponse response, long lIntakeID)
    {
        CInstrumentResponse resp = new CInstrumentResponse(BaseMstr);
        return resp.RemoveResponse(BaseMstr.SelectedPatientID, response.MID, lIntakeID, response.RID);
    }


    /// <summary>
    /// US:5966
    /// Class to model JSON object to pass on to the callback 
    /// of the load instrument DirectMethod
    /// </summary>
    public class InstrumentOptions {
        public InstrumentOptions() { }
        public string SkipPattern { set; get; }
        public string TextMask { set; get; }
        public string ResponsesData { set; get; }
        public bool ReadOnly { set; get; }
    }

    public class AddInstrumentResponsesResult {
        public bool saved { set; get; }
        public string patient_id { set; get; }
        public long mid { set; get; }
        public long intake_id { set; get; }
        public long cpa_id { set; get; }
        public long pat_pw_id { set; get; }
        public long pat_pw_event_id { set; get; }
        public long pw_event_module_id { set; get; }
        public bool score_error { set; get; }
        public string score_error_msg { set; get; }
        public string title { set; get; }
        public string message { set; get; }
        public long status_code { set; get; }
    }


    private class JScoreRow
    {
        public long series { set; get; }
        public decimal score { set; get; }
        public string interpretation { set; get; }
        public string description {set; get; }
        public string severity { set; get; }
    }
}